#!/bin/bash

# Set environment variables
export TZ=America/Los_Angeles
export DEBIAN_FRONTEND=noninteractive

GITHUB_USERNAME=<YOUR_GITHUB_USERNAME>
GITHUB_PAT=<YOUR_GITHUB_PAT>
WANDB_API_KEY=<YOUR_WANDB_API_KEY>

# Function to check if micromamba is installed
function check_micromamba_installed {
    if command -v micromamba &> /dev/null
    then
        echo "Micromamba is already installed."
        return 0
    else
        return 1
    fi
}

# Install micromamba if not installed
if ! check_micromamba_installed; then
    echo "Micromamba not found, installing Micromamba..."
    
    # Download Micromamba installer
    wget https://micro.mamba.pm/install.sh -O ~/micromamba_install.sh
    
    # Run the installer
    bash ~/micromamba_install.sh -b -p $HOME/micromamba
    
    # Add micromamba to PATH
    echo "export PATH=\"$HOME/micromamba/bin:\$PATH\"" >> ~/.bashrc
    source ~/.bashrc
    
    echo "Micromamba installation completed."
else
    echo "Skipping Micromamba installation."
fi

rm ~/micromamba_install.sh # cleanup installer

# Initialize Micromamba
micromamba shell init -s bash -p $HOME/micromamba
source ~/.bashrc

# Add needed conda channels
micromamba config --add channels conda-forge
micromamba config --add channels nvidia/label/cuda-11.8.0

# Create a new micromamba environment with Python 3.10 named "supe"
micromamba create -n supe python=3.10 -y
micromamba activate supe

# Install basic development tools
micromamba install -y -c conda-forge cmake git curl jpeg libpng libtiff ffmpeg libffi \
    gtk3 glew openssl freeglut mesa-libgl-cos6-x86_64 mesalib

# Install additional utilities
micromamba install -y -c conda-forge wget gnupg

# Install CUDA and CUDNN
micromamba install -y -c nvidia cuda-toolkit=11.8
pip install nvidia-cudnn-cu11==8.9.6.50

# Install MuJoCo
mkdir -p $HOME/.mujoco
wget https://mujoco.org/download/mujoco210-linux-x86_64.tar.gz -P $HOME/.mujoco
tar -xvzf $HOME/.mujoco/mujoco210-linux-x86_64.tar.gz -C $HOME/.mujoco
rm $HOME/.mujoco/mujoco210-linux-x86_64.tar.gz

# Set environment variables for MuJoCo
echo 'export LD_LIBRARY_PATH=$LD_LIBRARY_PATH:/usr/lib/nvidia:$HOME/.mujoco/mujoco210/bin' >> ~/.bashrc
echo 'export XLA_PYTHON_CLIENT_PREALLOCATE=False' >> ~/.bashrc
source ~/.bashrc
micromamba activate supe

# GitHub credentials setup
git config --global credential.helper store
echo "https://${GITHUB_USERNAME}:${GITHUB_PAT}@github.com" > ~/.git-credentials

# Install Python dependencies
pip install --upgrade setuptools wheel
pip install -r requirements.txt
pip install "Cython<3"
pip install patchelf
pip install wandb

# Setup wandb
wandb login --relogin $WANDB_API_KEY

# Install JAX with CUDA support
pip install "jax[cuda11_pip]==0.4.16" -f https://storage.googleapis.com/jax-releases/jax_cuda_releases.html

# Clone and install d4rl
git clone https://github.com/Farama-Foundation/D4RL/tree/master
cd d4rl
pip install -e .
cd ..
